***********************************************************************
* Project: The Demand for Government Debt  – United Kingdom       *
* Description:                                                        *
* This script builds a quarterly panel dataset on UK government bond *
* holdings by sector, combined with macro-financial indicators,      *
* monetary policy shocks, and asset price series.                    *
***********************************************************************

clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/


//////////////////////////////////////////
//////// United Kingdom   ////////////////
//////////////////////////////////////////

//////////////////////////////////////////
//// ONS Data on Holdings  -  these are the ones being used in Motivation Graphs. 

import excel using "$path/raw_data/Gilt_holdings_data_updated.xlsx", sheet("Gilt_holdings_short") firstrow clear

rename L MFI
rename InsuranceCompaniesandPension ICPF
rename OtherFinancialInstitutions OFI
gen ROW=OverseasHoldingsRestofWorld
gen Centralbank=BankofEngland
egen HH=rowtotal(Households NonProfitInstitutionsserving )
egen AllOther=rowtotal(LocalGovernment PublicCorporations PrivateNonFinancialcompanies )

egen Total=rowtotal(ICPF MFI Centralbank ROW OFI HH AllOther)

foreach x of varlist ROW Centralbank MFI HH AllOther ICPF OFI Total {
replace `x'=`x'/1000000
}

gen year=substr(A,1,4)
destring year,replace

gen quarter=substr(A,7,1)
destring quarter,replace

gen yq = yq(year, quarter)
format yq %tq


//////////////////////////////////////////
// Merge with FRED data - Get VIX, GDP
/*
preserve

set fredkey 2f2fb8729294c4714471cd4d993ad7f7,perm
	import fred VIXCLS UKNGDP THREEFY5 THREEFY8, daterange(1966-01-01 2024-09-23) aggregate(quarter,eop) clear
		
	gen yq=qofd(daten)	

	tsset yq
	
	gen GDP=UKNGDP*4
	
	tempfile fred
	save `fred',replace
	save "$path/generated_data/UK_FRED.dta",replace
	*saved on July 17, 2025.
restore

*merge m:1 yq using `fred',nogen	
*/
merge m:1 yq using "$path/generated_data/UK_FRED.dta",nogen

//////////////////////////////////////////



//////////////////////////////////////////
// Merge with BBG data - Get Yields, FTSE 100 Return - Get this data from BBG (Alexis folder)

preserve 
import excel using "$path/raw_data/DBSonline_BBG_UK_updated.xlsx", firstrow clear sheet("data") 

rename UKGilts3Month GRUK3M
rename UKGilts10Yr GUKG10
rename UKGilts1Year GUKG1
rename UKGilts20Year GUKG20
rename UKGilts2Year GUKG2
rename UKGilts30Year GUKG30
rename UKGilts5Year GUKG5

rename GBPUSDBS3MV3M1Y USDGBPCIP1y
rename FTSE100INDEX  FTSE100
rename USDGBPXRATE usdgbp

gen yq=qofd(A)	

collapse G* USDGBPCIP1y usdgbp FTSE100,by(yq)

tsset yq

gen ftse_ret=(ln(FTSE100)-ln(l.FTSE100))*100
replace USDGBPCIP1y = -USDGBPCIP1y/100

keep yq G* USDGBPCIP1y usdgbp FTSE100 ftse_ret

tempfile uk_bbg
save `uk_bbg'
restore

merge 1:1 yq using `uk_bbg',nogen

//////////////////////////////////////////
// Merge with Bank of England assets (in USD - convert to gbp) - Get this data from BIS Economic Dashboards (DBSOnline)

preserve 

import excel using "$path/raw_data/DBSonline_BoEassets_UK_updated.xlsx", firstrow clear sheet("data")

gen date=date(A,"MDY")

gen year=yofd(date)
gen quarter=quarter(date)

gen yq=yq(year,quarter)
format yq %tq

rename BoE_totalassets BoE_usd
keep yq BoE_usd

tempfile boe
save `boe'
restore

merge 1:1 yq using `boe',nogen

gen BoE_gbp=BoE*usdgbp


//////////////////////////////////////////

//////////////////////////////////////////
// Merge with Macro data - Get Debt to GDP, GDP, CPI - Get this data from BIS Economic Dashboards (DBSOnline)

preserve
import excel using "$path/raw_data/DBSonline_MACRO_UK_updated.xlsx", firstrow clear sheet("data") cellrange(A2:N440)

rename BIS_MACROQVEBAGBL1 CPI

gen yq=qofd(A)
format yq %tq

tsset yq

gen inf_uk=(ln(CPI)-ln(l.CPI))*100

keep yq  CPI inf_uk

tempfile uk_macro
save `uk_macro'
restore

merge 1:1 yq using `uk_macro',nogen

//////////////////////////////////////////
// Debt-to-GDP

preserve

import excel using "$path/raw_data/series-260924.xls", clear cellrange(A256:B633)

local y=tm(1993m3)

gen ym=`y'+_n-1
format ym %tm
gen yq=qofd(dofm(ym))
format yq %tq
bys yq: gen h=_n
bys yq: gen H=_N
keep if h==H
tsset yq
keep yq B

rename B debt_gdp_uk

tempfile dguk
save `dguk'
restore

merge 1:1 yq using `dguk',nogen

*gen debt_gdp_uk= (Total/GDP *1000000) *100

//////////////////////////////////////////

// ZC rates
preserve
import excel using "$path/raw_data/data_ZCYield.xlsx", firstrow clear sheet("GB") 
gen yq = qofd(dofc(Date))

format yq %tq

bys yq: gen h=_n
bys yq: gen H=_N
keep if h==H

tsset yq
drop Date

tempfile zc
save `zc'
restore

merge 1:1 yq using `zc',nogen

//////////////////////////////////////////


// ADD UK MP Shocks from Measuring monetary policy in the UK: The UK monetary policy event-study database by Robin Braun, Silvia Miranda-Agrippino, Tuli Saha ; https://www.sciencedirect.com/science/article/abs/pii/S0304393224000989

preserve

import excel using "$path/raw_data/MP_shocks_other", firstrow clear sheet("BOE2_Q") 
gen yq = yq(Year, Quarter)
format yq %tq

keep yq Target Path QE

order yq Target Path QE

rename Target BOE_BMAS_Target
rename Path BOE_BMAS_Path
rename QE BOE_BMAS_QE

tempfile boe_bmas
save `boe_bmas',replace

restore

merge 1:1 yq using `boe_bmas',nogen


tsset yq

save "$path/generated_data/UK_data_v202507.dta", replace





/*
// Volumes
gen a_1=Centralbank
gen a_2=a_1+ MFI
gen a_3=a_2 + ROW
gen a_4=a_3 + ICPF
gen a_5=a_4 + OFI
gen a_6=a_5 + HH
gen a_7=a_6 + AllOther

label var a_1 "Central bank"
label var a_2 "MFIs"
label var a_3 "ROW"
label var a_4 "ICPF"
label var a_5 "Other FI"
label var a_6 "HH" 
label var a_7 "All other"
/*
gen a_1=Centralbank 
gen a_2=a_1 + ROW
gen a_3=a_2 + ICPF
gen a_4=a_3 + OFI
gen a_5=a_4 + MFI
gen a_6=a_5 + HH
gen a_7=a_6 + AllOther

label var a_1 "Central bank"
label var a_2 "ROW"
label var a_3 "ICPF"
label var a_4 "Other FI"
label var a_5 "MFIs"
label var a_6 "HH" 
label var a_7 "All other"
*/

cap gen Covid=2.6
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}

cap gen GFC=2.6
foreach i of numlist `=tq(2008q3)' {
	cap gen y`i'=`i'
}

local y=tq(2000q1)
twoway   (area a_7 yq, bcolor(black*0.1))  (area a_6 yq, bcolor(black*0.2)) (area a_5 yq, bcolor(black*0.3)) ///
  (area a_4 yq, bcolor(black*0.4))  (area a_3 yq, bcolor(black*0.6)) (area a_2 yq, bcolor(black*0.7)) ///
  (area a_1 yq, bcolor(black)) (dropline GFC y`=tq(2008q3)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  (dropline Covid y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  if year>=`y', ytitle("£ trillions") xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
  legend(rows(2) order(7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(5) size(*.9)  symxsize(*.4))  ///
  text(2.7 `=tq(2020q1)' "Covid-19") text(2.7 `=tq(2008q3)' "GFC") ylabel(0(0.5)2.7)
graph export "$path/Data/analysis/graphs_paper/UK_volumes.pdf",replace


// Shares
foreach x of varlist Centralbank ROW ICPF MFI OFI HH AllOther {
gen shr_`x'=(`x'/Total)*100
}

// Note: the graph looks funny because of negative positions of MFIs pre-GFC. That is due to their market making/prop trading activities - need a footnote to explain this. 

gen b_1= shr_Centralbank
gen b_2=b_1 + shr_MFI
gen b_3=b_2 + shr_ROW
gen b_4=b_3 + shr_ICPF
gen b_5=b_4 + shr_OFI
gen b_6=b_5 + shr_HH
gen b_7=b_6 + shr_AllOther

label var b_1 "Central bank"
label var b_2 "MFIs"
label var b_3 "ROW"
label var b_4 "ICPF"
label var b_5 "Other FI"
label var b_6 "HH" 
label var b_7 "All other"
/*
gen b_1=shr_Centralbank 
gen b_2=b_1 + shr_ROW
gen b_3=b_2 + shr_ICPF
gen b_4=b_3 + shr_OFI
gen b_5=b_4 + shr_MFI
gen b_6=b_5 + shr_HH
gen b_7=b_6 + shr_AllOther

label var b_1 "Central bank"
label var b_2 "ROW"
label var b_3 "ICPF"
label var b_4 "Other FI"
label var b_5 "MFIs"
label var b_6 "HH" 
label var b_7 "All other"
*/
cap gen Covid_2=100
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}

cap gen GFC_2=100
foreach i of numlist `=tq(2008q3)' {
	cap gen y`i'=`i'
}


local y=tq(2000q1)
twoway  (area b_7 yq, bcolor(black*0.1)) (area b_6 yq, bcolor(black*0.2)) (area b_5 yq, bcolor(black*0.3)) ///
 (area b_4 yq, bcolor(black*0.4))  (area b_3 yq, bcolor(black*0.6)) (area b_2 yq, bcolor(black*0.7)) ///
  (area b_1 yq, bcolor(black)) (dropline GFC_2 y`=tq(2008q3)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  (dropline Covid_2 y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  if year>=`y', ytitle("Per cent" " " ) xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
  legend(rows(2) order(7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(5) size(*.9)  symxsize(*.4))  ///
  text(103 `=tq(2020q1)' "Covid-19") text(103 `=tq(2008q3)' "GFC") 
graph export "$path/Data/analysis/graphs_paper/UK_shares.pdf",replace


// Regressions
tsset yq
gen pre_gfc=(year<=2008)
gen post_gfc=(year>=2009 & year<=2019)
gen post_covid=year>=2020

foreach x of varlist Centralbank ROW ICPF MFI HH OFI AllOther Total {
	
	gen d`x'=d.`x'
}

foreach x of varlist Centralbank ROW ICPF MFI HH OFI AllOther Total {
	
		gen d`x'_n = d`x'/l.Total
}

gen tot_beforegfc=dTotal_n*pre_gfc
gen tot_pgfc=dTotal_n*post_gfc
gen tot_pc=dTotal_n*post_covid


**** Marginal holders of UK Gilts

local format = "nocons nonotes dec(2) label"

local reg_table = "$path/Data/analysis/tables_paper/UK_marginal_holders"
cap erase "`reg_table'.tex"
cap erase "`reg_table'.txt"

ivreg2 dCentralbank_n tot_beforegfc tot_pgfc tot_pc ,robust bw(auto)

outreg2 using "`reg_table'.tex", tex(frag) replace `format' 

foreach x of varlist dROW_n dICPF_n dMFI_n dHH_n dOFI_n dAllOther_n  {
	
ivreg2 `x' tot_beforegfc tot_pgfc tot_pc ,robust bw(auto)
outreg2 using "`reg_table'.tex", tex(frag) append `format'

}

cap erase "`reg_table'.txt"
