***********************************************************************
* Project: The Demand for Government Debt – United States        *
* Description:                                                        *
* This script builds a comprehensive quarterly dataset on the U.S.    *
* Treasury market for use in regression analysis. It combines Flow of *
* Funds (FoF) holdings data, TIC foreign holdings, macro-financial    *
* indicators, monetary policy shocks, valuation adjustments, and      *
* newly added Treasury refunding shock series.                        *
***********************************************************************

clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/

//////////////////////////////////////////
//////// United States   /////////////////
//////////////////////////////////////////
/*
// Run this once to get the FoF data from Fred - Fed sometimes revises these data, so the results might not exactly replicate if it is retrieved any other time. This version is retrieved on 8.Oct.2024. 

import excel using "$path/raw_data/Data.xlsx", sheet("Treasury holders update") firstrow clear

keep if A=="date"

rename * i_*
rename i_A A

tostring *, replace 
reshape long i_,i(A) j(sector) string
drop if i_=="."

generate str fred = "BOGZ1" + substr(i_, 1, strlen(i_) - 2) + substr(i_, 13, strlen(i_) )
replace fred = "BOGZ1FL213061103Q" if sector=="Stateandlocalgovernment"  // FRED discontinued the previous series. 

preserve
keep fred sector
save "$path/generated_data/AUX_FoF.dta",replace
restore

gen a = 1
gen b=fred
keep fred a b 
reshape wide fred, i(a) j(b) string
rename fred* *
drop a
ds
set fredkey 2f2fb8729294c4714471cd4d993ad7f7,perm
import fred `r(varlist)', daterange(1966-01-01 2024-09-23) aggregate(quarter,eop) clear

drop datestr

reshape long BOGZ1, i(daten) j(fred) string

replace fred= "BOGZ1" + fred

merge m:1 fred using "$path/generated_data/AUX_FoF.dta",nogen

drop fred
rename BOGZ1 b

reshape wide b, i(daten) j(sector) string

rename b* *

egen Banks=rowtotal(USchartereddepositoryinstitut ForeginbankingofficesinUS BanksinUSaffiliatedareas Creditunions )
egen Pensionfund = rowtotal(Privatepensionfund Federalgovernmentretirementfu Stateandlocalgovernmentretir )
egen Insurancecompanies= rowtotal(PropertyCasualtyinsurancecomp Lifeinsurancecompanies)

// Data originally in millions, this converts it to trillions.
ds daten,not
foreach x of varlist `r(varlist)' {
replace `x'=`x'/1000000
}

gen yq=qofd(daten)
format yq %tq

gen year=yofd(daten)


//////////////////////////////////////////
/// Different countries - UST holdings

preserve

import excel using "$path/raw_data/USTHoldings_20240924.xlsx", sheet("CountryData") firstrow clear

format A %td
gen yq=qofd(A)
format yq %tq

*rename HOLDTFOIndex row_official
*rename Private row_private

keep yq HOLDBEIndex HOLDIRIndex HOLDSWIndex HOLDCAYMIndex HOLDOPECIndex HOLDFRIndex HOLDGEIndex HOLDJNIndex HOLDCHIndex HOLDUKIndex HOLDTFOIndex HOLDTOTIndex
rename *Index *

// March 2016 data is missing for OPEC
replace HOLDOPEC=. if yq>=tq(2016q2)


foreach x of varlist HOLDBE HOLDIR HOLDCAYM {
destring `x',replace force
}

egen HOLDDUM=rowtotal(HOLDBE HOLDIR HOLDSW HOLDCAYM HOLDOPEC HOLDFR HOLDGE HOLDJN HOLDCH HOLDUK)
gen HOLDOTHER = HOLDTOT - HOLDDUM

rename HOLDTFO row_fo
gen row_fp=HOLDTOT - row_fo
drop HOLDDUM HOLDTOT

collapse HOLD* row*, by(yq)

rename row_fo bn_row_fo
rename row_fp bn_row_fp

gen row_fo=bn_row_fo/1000
gen row_fp=bn_row_fp/1000

tempfile row
save `row'
restore

merge 1:1 yq using `row',nogen
//////////////////////////////////////////

order yq HH Nonfinancialcorporate Nonfinancialnoncorporate Stateandlocalgovernment Monetaryauthority Banks Insurancecompanies Pensionfund MMFs Mutualfunds Closedendfunds ETFs GSE ASissuers Brokersanddealers Holdingcompanies Otherfinancialbusiness row_fo row_fp ROW 

egen Total_new=rowtotal(HH-row_fp)

gen ICPF = Pensionfund + Insurancecompanies 
rename Pensionfund  PF
rename Insurancecompanies  IC
gen IF = Mutualfunds + Closedendfunds + ETFs
gen SLG=Stateandlocalgovernment 
gen Other=Total_new - ICPF - IF - SLG - Monetaryauthority - HH - row_fo - row_fp - MMFs  - Banks


// Create all non-CB to be used in regressions
egen all_pri=rowtotal(Banks PF row_fo row_fp IF IC SLG HH Other)

save "$path/generated_data/FoF_fred_20240923.dta",replace
*/
//////////////////////////////////////////

use "$path/generated_data/FoF_fred_20240923.dta",clear

//////////////////////////////////////////
// Merge with FRED data

/*
preserve

set fredkey 2f2fb8729294c4714471cd4d993ad7f7,perm
	import fred BOGZ1FL313161205Q TREAST THREEFY10 THREEFY8 THREEFY5 DGS3MO DGS10 DGS5 DGS30 DGS2 DGS1 BAMLC0A0CM BAMLC0A1CAAA GDP VIXCLS USACPICORMINMEI WALCL DFF GFDEGDQ188S USREC FEDFUNDS IOER IORB A939RC0Q052SBEA NFCI STLFSI4, daterange(1966-01-01 2024-09-23) aggregate(quarter,eop) clear
		
	gen yq=qofd(daten)	

	tsset yq
	
	gen old_UST_fv=BOGZ1FL313161205Q/1000000 // this series is discontinued, get treasury data below from the treasury website. 
	gen SOMA_fv=TREAST/1000000
	gen inf_exc_fe=(ln(USACPICORMINMEI)-ln(l.USACPICORMINMEI))*100
	*gen stock_ret=(ln(WILL4500IND)-ln(l.WILL4500IND))*100 //FRED discontinued it. 
	rename DFF ffr
	rename GFDEGDQ188S debt_gdp_us
	rename A939RC0Q052SBEA gdp_per_capita
	
	
	replace IORB=IOER if missing(IORB)
	
	rename BAMLC0A0CM spread_ig
	rename BAMLC0A1CAAA spread_aaa
	
	tempfile fred
	save `fred',replace
	save "$path/generated_data/US_FRED_macrofin.dta",replace
	* saved on July 16, 2025
restore

merge m:1 yq using `fred',nogen	
*/

merge m:1 yq using "$path/generated_data/US_FRED_macrofin.dta",nogen
//////////////////////////////////////////
gen old_fv_all_pri=old_UST_fv - SOMA_fv - MMFs

preserve
import excel using "$path/raw_data/MSPD_SumSecty_20010131_20241031.xlsx",clear firstrow
keep if SecurityTypeDescription=="Total Marketable"

gen yq=qofd(RecordDate)

gen UST_fv=DebtHeldbythePublicinMill/1000000

bys yq: egen latest=max(RecordDate)
keep if RecordDate==latest

keep yq UST_fv

tempfile ust
save `ust'
restore

merge 1:1 yq using `ust',nogen

gen fv_all_pri=UST_fv - SOMA_fv - MMFs


/// BRW Shocks

preserve 
*insheet using "$path/raw_data/BRW shock series 2021Dec.csv",clear
import excel using "$path/raw_data/BRW2024-08.xlsx",clear cellrange(A2:B369) firstrow

rename Month month
rename BRW brw_monthly

gen year=substr(month,1,4)
gen mon=substr(month,6,.)
destring year mon, replace
gen ym=ym(year,mon)
gen yq=qofd(dofm(ym))
format yq %tq
collapse (sum) brw_monthly , by(yq)
rename brw_monthly pct_brw
gen brw_monthly=pct_brw/100

format yq %tq

tempfile brw
save `brw'
restore
merge 1:1 yq using `brw'

/////////////////


/// Swanson shocks

preserve
import excel using "$path/raw_data/Swanson_shocks.xlsx", sheet("Data") firstrow clear

gen date=date(B, "MDY")
gen yq=qofd(date)
format yq %tq

collapse (sum) FederalFundsRatefactor ForwardGuidancefactor LSAPfactor,by(yq)

tempfile swanson
save `swanson',replace
restore

merge 1:1 yq using `swanson',nogen
/////////////////////////////////////


/// Jarocinski-Karadi shocks

preserve

*insheet using "$path/raw_data/jarocinski_karadi_shocks.csv",clear
import excel using "$path/raw_data/shocks_fed_jk_m_202409.xlsx",  firstrow clear

gen ym=ym(year,month)

gen yq=qofd(dofm(ym))

rename MP_median mp_median
rename CBI_median cbi_median

collapse (sum) mp_median cbi_median,by(yq)
format yq %tq
** Org data is in percentages. 
*replace mp_median=mp_median/100
*replace cbi_median=cbi_median/100

tempfile jk
save `jk',replace
restore

merge 1:1 yq using `jk',nogen

///////////////////////////


/////// Nakamura-Steinsson updated by Acosta

preserve
import excel using "$path/raw_data/MPshocksAcosta.xlsx", sheet("shocks") firstrow clear

gen yq=qofd(fomc)
collapse (sum) ns, by(yq)
format yq %tq

tempfile ns
save `ns',replace
restore

merge 1:1 yq using `ns',nogen


/// SP 500 Dividend Yield

preserve
import excel using "$path/raw_data/outside_assets_updated.xlsx", sheet("SP500 dividend yield") firstrow clear
rename A Date
format Date %td
gen yq=qofd(Date)
bys yq: gen day=_n
bys yq: gen eop=_N
keep if day==eop

rename SP500Dividendyield sp_div_yield

tempfile sp_div
save `sp_div',replace
restore

merge 1:1 yq using `sp_div',nogen

///////////////////////////

/// German yields

preserve 

import excel using "$path/raw_data/outside_assets_updated.xlsx", sheet("German zero coupon yield") firstrow cellrange(A6:D629) clear
drop C
rename B german_5y_zc
rename D german_10y_zc

gen date=date(A,"YM")
format date %td

gen yq=qofd(date)
gen month=month(date)
keep if inlist(month, 3,6,9,12)

keep yq german_5y_zc german_10y_zc


tempfile german
save `german',replace
restore

merge 1:1 yq using `german',nogen

// //////////////////////////


////// Rest of the portfolio
/*
preserve

set fredkey 2f2fb8729294c4714471cd4d993ad7f7,perm
	import fred BOGZ1LM564090005Q BOGZ1LM554090005Q BOGZ1LM654090000Q BOGZ1FL514090005Q BOGZ1FL544090005Q BOGZ1FL594090005Q BOGZ1FL764090005Q SLGTFFQ027S TFAABSHNO, daterange(1966-01-01 2024-09-23) aggregate(quarter,eop) clear
		
	gen yq=qofd(daten)	

	tsset yq
	
	rename TFAABSHNO HH_assets
	rename SLGTFFQ027S SLG_assets
	rename BOGZ1FL764090005Q Banks_assets
	rename BOGZ1FL594090005Q PF_assets
	gen IC_assets=BOGZ1FL544090005Q+BOGZ1FL514090005Q
	gen IF_assets= BOGZ1LM654090000 + BOGZ1LM554090005Q + BOGZ1LM564090005Q

	replace HH_assets=HH_assets/1000000
	replace SLG_assets=SLG_assets/1000000
	replace Banks_assets=Banks_assets/1000000
	replace PF_assets=PF_assets/1000000
	replace IF_assets=IF_assets/1000000
	replace IC_assets=IC_assets/1000000
	
	tempfile demand
	save `demand',replace
	save "$path/generated_data/US_FRED_FoF.dta",replace
	*saved on July 16, 2025.
restore

merge m:1 yq using `demand',nogen	
*/

merge m:1 yq using "$path/generated_data/US_FRED_FoF.dta",nogen

gen HH_exUST=HH_assets-HH
gen SLG_exUST=SLG_assets-SLG
gen Banks_exUST=Banks_assets-Banks
gen PF_exUST=PF_assets-PF
gen IF_exUST=IF_assets-IF
gen IC_exUST=IC_assets-IC



////// Outside asset as only debt securities
/*
preserve

set fredkey 2f2fb8729294c4714471cd4d993ad7f7,perm

/*
Mutual funds - debt securities: BOGZ1LM654022005Q
Closed end funds- debt securities: BOGZ1LM554022005Q
ROW - debt securities: ROWDASQ027S
SLG - debt securities: SLGDASQ027S
ETF - debt securities: BOGZ1LM564022005Q
PF - debt securities: BOGZ1LM594022005Q
IC - debt securities: BOGZ1LM544022005Q + BOGZ1LM514022005Q
Banks - debt securities: BOGZ1LM704022005Q


*/

	import fred BOGZ1LM654022005Q BOGZ1LM554022005Q ROWDASQ027S SLGDASQ027S BOGZ1LM564022005Q  BOGZ1LM594022005Q BOGZ1LM544022005Q BOGZ1LM514022005Q BOGZ1LM704022005Q, daterange(1966-01-01 2024-09-23) aggregate(quarter,eop) clear
	
	gen yq=qofd(daten)	

	tsset yq
	
	gen IF_debt= BOGZ1LM654022005Q + BOGZ1LM554022005Q + BOGZ1LM564022005Q
	rename SLGDASQ027S SLG_debt
	rename ROWDASQ027S ROW_debt 
	rename BOGZ1LM704022005Q Banks_debt
	rename BOGZ1LM594022005Q PF_debt
	gen IC_debt= BOGZ1LM544022005Q + BOGZ1LM514022005Q
	

	replace SLG_debt=SLG_debt/1000000
	replace Banks_debt=Banks_debt/1000000
	replace PF_debt=PF_debt/1000000
	replace IF_debt=IF_debt/1000000
	replace IC_debt=IC_debt/1000000
	replace ROW_debt = ROW_debt/1000000
	
	tempfile demand_debt
	save `demand_debt',replace
	save "$path/generated_data/US_FRED_FoF_debt.dta",replace
	
restore

merge m:1 yq using `demand_debt',nogen	
*/

merge m:1 yq using "$path/generated_data/US_FRED_FoF_debt.dta",nogen

egen tot_ROW=rowtotal(row_fo row_fp)
gen pri_shr=row_fp/tot_ROW


gen SLG_debtoa=SLG_debt-SLG
gen Banks_debtoa=Banks_debt-Banks
gen PF_debtoa=PF_debt-PF
gen IF_debtoa=IF_debt-IF
gen IC_debtoa=IC_debt-IC
gen row_fp_debtoa=ROW_debt*pri_shr-row_fp


preserve 
import excel "$path/raw_data/IMFCOFER.xlsx",clear firstrow
gen yq = quarterly(period, "YQ")
format yq %tq

rename QuarterlyAllCountriesexclu dollar_share

keep yq dollar_share
tempfile row_fo
save `row_fo'
restore

merge 1:1 yq using `row_fo',nogen

//////////////////////////////////

tsset yq

save "$path/generated_data/US_data_v202507.dta", replace

