***********************************************************************
* Project: The Demand for Government Debt – Japan                    *
* Description:                                                       *
* This script constructs a dataset for demand elasticity analysis of *
* Japanese government bonds. It merges BOJ monetary policy shocks,   *
* macro variables, and sectoral holdings. It computes valuation-     *
* adjusted holdings, derives PCA-based instruments, and estimates    *
* elasticities across institutional sectors using 2SLS regressions.  *
* Outputs include F-stat tables, 2SLS results, and time series plots.*
***********************************************************************

clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/


////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

// Clean and transform variables for elasticity regressions
use "$path/generated_data/mon_pol_shocks_v202507.dta",clear
merge 1:1 yq using "$path/generated_data/broad_nominal_FX_v202507.dta",nogen
merge 1:1 yq using "$path/generated_data/JP_data_v202507.dta",nogen

tsset yq

//Generate logs and log changes

// Holding data
foreach x of varlist Banks ICPF Publicpensions HH ROW Others {
gen dln_`x'=ln(`x')-ln(l.`x')
gen ln_`x'=ln(`x')
}

// Valuation changes - assuming everyone holds X-year zero coupon bonds.

gen val_ch10_pct=-(d.JP_10Y/100)*10
gen val_ch8_pct=-(d.JP_8Y/100)*8
gen val_ch5_pct=-(d.JP_5Y/100)*5




foreach x of varlist Banks ICPF Publicpensions HH ROW Others {
	foreach i in 5 8 10 {
gen val_ch`i'_`x'= l.`x'*val_ch`i'_pct
gen `x'_exval`i'=`x'-val_ch`i'_`x'
gen ln_`x'_exval`i'=ln(`x'_exval`i')
gen dln_`x'_exval`i'=ln(`x'_exval`i')-ln(l.`x'_exval`i')
}
}
///////


foreach y of varlist GDP CPI VIXCLS NBJP  {
gen dln_`y'=ln(`y')-ln(l.`y')
gen ln_`y'=ln(`y')
}


// Interest rates
*foreach t of varlist THREE* DGS* inf_exc_fe stock_ret usd_ret ffr spread_ig spread_aaa sp_div_yield german_5y_zc german_10y_zc {
	foreach t of varlist JP_* inf_jp nikkei_ret jpy_ret THREEFY5 THREEFY8 {

gen dir_`t'=`t'/100-l.`t'/100
gen ir_`t'=`t'/100
gen dln_`t'=ln(1+`t'/100)-ln(1+l.`t'/100)
gen ln_`t'=ln(1+`t'/100)

}



// Mon pol shocks
foreach t of varlist Ch* {
gen dln_`t'=ln(1+`t')-ln(1+l.`t')
gen ln_`t'=ln(1+`t')
gen dir_`t'=`t'-l.`t'
gen ir_`t'=`t'
}


// Principal components
tsset yq

pca BOJ_KS*
predict p1_boj_ks

pca ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ
predict p1_boj_ksx

pca BOJ_KS* ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ
predict p1_boj_ks_ksx

foreach mps in "p1_boj_ksx" "p1_boj_ks_ksx" "p1_boj_ks" {
egen std_`mps'=std(`mps')
}

global ln_controls "ln_GDP dln_GDP ir_inf_jp ln_NBJP ln_VIXCLS ir_nikkei_ret ir_THREEFY8 yq yq_sq" 

gen yq_sq=yq^2


label var ir_JP_8Y "JP 8Y Yield (ZC)"
label var std_p1_boj_ks "PCA 1 BOJ (KS)"
label var std_p1_boj_ks_ksx "PCA 1 BOJ (KS, KSX)"
label var std_p1_boj_ksx "PCA 1 BOJ (KSX)"
label var BOJ_KS_Target "BOJ KS Target"
label var BOJ_KS_Path "BOJ KS Path"
label var ir_Ch3m_BoJ "BOJ KSX 3m"
label var ir_Ch2y_BoJ "BOJ KSX 2y"
label var ir_Ch10y_BoJ "BOJ KSX 10y"


label var ln_Banks_exval8 "log(Banks)"
label var ln_ICPF_exval8 "log(ICPF)"
label var ln_Publicpensions_exval8 "log(PP)"
label var ln_ROW_exval8 "log(ROW)"
label var ln_HH_exval8 "log(HH)"
label var ln_Others_exval8 "log(Others)"


/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////        FIRST STAGE REGRESSIONS      //////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////// FIRST STAGE ///////////////////////////////

/// Create locals for the effective F-stats for various instruments

local mp1 BOJ_KS_Target BOJ_KS_Path
local text1 = "addtext(Trend, Yes, Controls, No, Sample, 2005q2-2020q1)"
local text1_wc = "addtext(Trend, Yes, Controls, Yes, Sample, 2005q2-2020q1)"

local mp2 ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ
local text2 = "addtext(Trend, Yes, Controls, No , Sample, 2005q2-2024q2)"
local text2_wc = "addtext(Trend, Yes, Controls, Yes , Sample, 2005q2-2024q2)"

local mp3 std_p1_boj_ks
local text3 = "addtext(Trend, Yes, Controls, No, Sample, 2005q2-2020q1)"
local text3_wc = "addtext(Trend, Yes, Controls, Yes, Sample, 2005q2-2020q1)"

local mp4 std_p1_boj_ksx
local text5 = "addtext(Trend, Yes, Controls, No , Sample, 2005q2-2024q2)"
local text5_wc = "addtext(Trend, Yes, Controls, Yes , Sample, 2005q2-2024q2)"

local mp5 std_p1_boj_ks_ksx
local text4 = "addtext(Trend, Yes, Controls, No, Sample, 2005q2-2020q1)"
local text4_wc = "addtext(Trend, Yes, Controls, Yes, Sample, 2005q2-2020q1)"




forvalues i=1/5 {
	
	// Without any controls
	*qui ivreg2 ln_Banks_exval5 (ir_JP_7Y=`mp`i'') yq yq_sq if yq>tq(2005q1),robust bw(auto)  ffirst
	*qui weakivtest
	
	*local weak`i' = `r(F_eff)'
	*local crt`i'=`r(c_TSLS_10)'
	
	// With controls
	qui ivreg2 ln_Banks_exval8 (ir_JP_8Y=`mp`i'') $ln_controls if yq>tq(2005q1),robust bw(auto)  ffirst
	qui weakivtest
	
	local weak`i'_wc = `r(F_eff)'
	local crt`i'_wc=`r(c_TSLS_10)'
	
	local format_fs`i' = "label addstat(Effective F-stat, `weak`i'', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'') adec(2) nonotes nocons keep(BOJ_KS_Target BOJ_KS_Path ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ std_p1_boj_ks  std_p1_boj_ksx std_p1_boj_ks_ksx) sortvar(BOJ_KS_Target BOJ_KS_Path ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ std_p1_boj_ks  std_p1_boj_ksx std_p1_boj_ks_ksx)"
	
	local format_fs`i'_wc= "label addstat(Effective F-stat, `weak`i'_wc', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'_wc') adec(2) nonotes nocons keep(BOJ_KS_Target BOJ_KS_Path ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ std_p1_boj_ks  std_p1_boj_ksx std_p1_boj_ks_ksx) sortvar(BOJ_KS_Target BOJ_KS_Path ir_Ch3m_BoJ ir_Ch2y_BoJ ir_Ch10y_BoJ std_p1_boj_ks  std_p1_boj_ksx std_p1_boj_ks_ksx)"

}


//// First stage table -  BASELINE

local table = "$path/tables_paper/MaSc_JP_pca_fs"
cap erase "`table'.tex"
cap erase "`table'.txt"

/*
forvalues i=1/3 {
	
		ivreg2 ir_JP_7Y `mp`i''  yq yq_sq if yq>tq(2005q1), robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format_fs`i'' dec(4) `text`i''

}
*/

forvalues i=1/5 {
	
		ivreg2 ir_JP_8Y `mp`i'' $ln_controls if yq>tq(2005q1), robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format_fs`i'_wc' dec(4) `text`i'_wc'

}


cap erase "`table'.txt"


// Second stage - Baseline: Using only KSX due to longer sample & comparability with the US

// Only KSX 

local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_JP_8Y) keep(ir_JP_8Y) "
*local format_overid = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp), Hansen J stat (p-val), e(jp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

*local format_wc = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8)"

local text_ss1 = "addtext(Trend, Yes, Controls,  Yes)"

local table = "$path/tables_paper/MaSc_JP_pca_2sls_ksx"
cap erase "`table'.tex"
cap erase "`table'.txt"

/// Second stage table - BASELINE

*foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_HH_exval8 ln_SLG_exval8 ln_Other_exval8 {
	
foreach x of varlist ln_ROW_exval8  ln_Banks_exval8 ln_HH_exval8 ln_ICPF_exval8 ln_Publicpensions_exval8 ln_Others_exval8 {

ivreg2 `x' (ir_JP_8Y=std_p1_boj_ksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

}

cap erase "`table'.txt"

// KS_KSX 
local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_JP_8Y) keep(ir_JP_8Y) "
*local format_overid = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp), Hansen J stat (p-val), e(jp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

*local format_wc = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8)"

local text_ss1 = "addtext(Trend, Yes, Controls,  Yes)"

local table = "$path/tables_paper/MaSc_JP_pca_2sls_ks_ksx"
cap erase "`table'.tex"
cap erase "`table'.txt"

*foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_HH_exval8 ln_SLG_exval8 ln_Other_exval8 {
	
foreach x of varlist ln_ROW_exval8  ln_Banks_exval8 ln_HH_exval8 ln_ICPF_exval8 ln_Publicpensions_exval8 ln_Others_exval8 {

ivreg2 `x' (ir_JP_8Y=std_p1_boj_ks_ksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

}

cap erase "`table'.txt"



// Time series of weighted elasticities


tsset yq

tempfile results_jp
local replace replace
foreach lhs in "ln_ROW_exval8" "ln_Banks_exval8" "ln_HH_exval8" "ln_ICPF_exval8" "ln_Publicpensions_exval8" "ln_Others_exval8"  {
	
		xi: ivreg2 `lhs' (ir_JP_8Y=std_p1_boj_ksx)  $ln_controls , robust bw(auto)
		regsave using "`results_jp'", pval autoid `replace' addlabel(lhs,"`lhs'") 
		local replace append
	
}

use "`results_jp'", clear
list

gen sector=substr(lhs,4,length(lhs)-10)
keep if inlist(var,"ir_JP_8Y") 
*replace var=substr(var,9,.)
preserve
use "$path/generated_data/JP_data_v202507.dta",clear
egen max_yq=max(yq) if !missing(Banks)
keep if yq==max_yq
keep ROW Banks HH ICPF Publicpensions Others 
xpose, clear v
rename _varname sector
rename v1 latest_holdings
*drop v2
tempfile hold
save `hold'
restore

merge m:1 sector using `hold',nogen

sort sector var

gen coef_95_low=coef-1.96*stderr 
gen coef_95_up=coef+1.96*stderr 

gen coef_90_low=coef-1.64*stderr 
gen coef_90_up=coef+1.64*stderr 


set obs 6
replace sector="Centralbank" if _n==6
*replace sector="Other" if _n==10

replace coef=0 if inlist(sector,"Centralbank")

keep sector coef

preserve
use "$path/generated_data/JP_data_v202507.dta", clear
*drop if HH==.

keep if inrange(yq,tq(2005q1),tq(2024q2))

keep ROW Banks HH ICPF Others Publicpensions Centralbank yq

ds yq,not
foreach i in `r(varlist)' {
	rename `i' hold`i'
}

reshape long hold,i(yq) j(sector,s)


tempfile holdings
save `holdings'
restore

merge 1:m sector using `holdings',nogen

*drop if sector=="Other"
*drop if sector=="MMFs"
drop if sector=="Centralbank"
encode sector,gen(sector_enc)

xtset sector_enc yq

bys yq: egen Total_actual=sum(hold)
gen shr_act=hold/Total_actual
gen wgt_coef=coef*shr_act
bys yq: egen mkt_elas_yq_actual=sum(wgt_coef)

gen impact_actual=(1/mkt_elas_yq_actual)*100

tw (line mkt_elas_yq_actual yq,lwidth(vthick) lcolor(black)) 
tw (line impact_actual yq,lwidth(vthick) lcolor(black)) , scheme(stcolor) legend(pos(6) rows(2) symysize(*0.5) size(small)) xtitle(" ") ytitle("Impact on yields of 1% higher residual supply (bps)")
graph export "$path/graphs_paper/JP_yield_impact.pdf",replace 

