***********************************************************************
* Project: The Demand for Government Debt – United States            *
* Description:                                                       *
* This script runs regression analyses to study the demand for       *
* U.S. government debt across investor sectors and policy regimes.   *
* It evaluates the responsiveness of holdings to interest rates,     *
* using various monetary policy indicators and interaction terms.    *
***********************************************************************

clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/

use "$path/generated_data/us_regression_cf_v202409.dta",clear

*drop if year>=2024
keep if inrange(yq,tq(2004q3),tq(2024q2))

/////////////////////////////////////////////
/////////////////////////////////////////////
//// Regression tables 
/////////////////////////////////////////////
/////////////////////////////////////////////

tsset yq


global ln_controls "ln_GDP dln_GDP ir_inf_exc_fe ln_NBUS ln_VIXCLS ir_german_5y_zc  yq yq_sq" 
global ln_controls_linear "ln_GDP dln_GDP ir_inf_exc_fe ln_NBUS ln_VIXCLS ir_german_5y_zc yq" 
global dln_controls "dln_GDP dir_inf_exc_fe dln_NBUS dir_german_5y_zc dln_VIXCLS" 
global ln_controls_many "ln_GDP dln_GDP inf_exc_fe ln_NBUS ln_UST_fv ln_VIXCLS ir_spread_aaa ir_german_5y_zc ir_sp_div_yield yq yq_sq"
global ds_controls "ln_GDP dln_GDP inf_exc_fe ln_NBUS yq yq_sq" 

global ln_controls_lag "l.ln_GDP l.dln_GDP l.ir_inf_exc_fe l.ln_NBUS l.ln_VIXCLS l.ir_german_5y_zc yq yq_sq" 
global ln_controls_novix "ln_GDP dln_GDP ir_inf_exc_fe ln_NBUS ir_german_5y_zc  yq yq_sq" 


label var ln_UST_fv "log UST (fv)"
label var dln_GDP "GDP growth"
label var inf_exc_fe "Inflation"
label var ln_VIXCLS "log VIX"
label var ir_german_5y_zc "5Y German yield (zc)"
label var ir_german_10y_zc "10Y German yield (zc)"
label var ln_NBUS "log broad dollar index"
label var yq "trend"
label var yq_sq "trend squared"

label var ir_DGS3MO "3MO Treasury yield"
label var ir_sp_div_yield "SP500 dividend yield"
label var ir_spread_ig "ICE BofA OAS IG"
label var ir_spread_aaa "ICE BofA OAS AAA"



/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////        BASELINE REGRESSIONS      //////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////// FIRST STAGE ///////////////////////////////

/// Create locals for the effective F-stats for various instruments

local mp1 std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor
local text1 = "addtext(Controls,  \bf X and W , Sample, 2004q3-2019q2)"
local text1_wc = "addtext(Sample, 1994q2-2019q2)"

local mp2 std_brw_monthly
local text2 = "addtext(Controls, \bf X and W  , Sample, 2004q3-2024q2)"
local text2_wc = "addtext(Sample, 1994q2-2024q2)"

local mp3 std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed
local text3 = "addtext(Controls, \bf X and W  , Sample, 2004q3-2024q2)"
local text3_wc = "addtext(Sample, 2004q3-2024q2)"

local mp4 std_mp_median 
local text4 = "addtext(Controls, \bf X and W  , Sample, 2004q3-2024q2)"
local text4_wc = "addtext(Sample, 1994q2-2024q2)"

local mp5 std_ns
local text5 = "addtext(Controls, \bf X and W , Sample, 2004q3-2022q3)"
local text5_wc = "addtext(Sample, 2004q3-2022q3)"

local mp6 std_p1_jkbrwksx
local text6 = "addtext(Controls, \bf X and W , Sample, 2004q3-2024q2)"
local text6_wc = "addtext(Sample, 2004q3-2024q2)"

local mp7 std_p1_jkbrwksxns
local text7 = "addtext(Controls,\bf X and W , Sample, 2004q3-2022q3)"
local text7_wc = "addtext(Sample, 2004q3-2022q3)"


forvalues i=1/7 {
	
		qui ivreg2 ln_Banks_exval8 (ir_THREEFY8=`mp`i'') $ln_controls ,robust bw(auto)  ffirst
qui weakivtest
	
	local weak`i' = `r(F_eff)'
	local crt`i'=`r(c_TSLS_10)'
	
	local format_fs`i' = "label addstat(Effective F-stat, `weak`i'', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'') adec(2) nonotes nocons keep(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns) sortvar(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns)"
	
	local format_fs`i'_wc= "label addstat(Effective F-stat, `weak`i'', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'') adec(2) nonotes nocons sortvar(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns)"

}

forvalues i=1/7 {
	
qui ivreg2 diff_Banks (ir_THREEFY8=`mp`i'') $ds_controls ,robust bw(auto)  ffirst
qui weakivtest

	local weak`i'_ds = `r(F_eff)'
	local crt`i'_ds=`r(c_TSLS_10)'
	local format_fs`i'_ds = "label addstat(Effective F-stat, `weak`i'_ds', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'_ds') adec(2) nonotes nocons keep(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns) sortvar(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns)"
	local format_fs`i'_wc_ds= "label addstat(Effective F-stat, `weak`i'_ds', Crt. Val. \$ \alpha=5%$ and \$ \tau=10%$, `crt`i'_ds') adec(2) nonotes nocons sortvar(std_FederalFundsRatefactor std_ForwardGuidancefactor std_LSAPfactor  std_brw_monthly std_ir_Ch3m_Fed std_ir_Ch2y_Fed std_ir_Ch10y_Fed  std_mp_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns)"
	
}

//// MAIN TEXT - First stage table 

local table = "$path/tables_paper/MaSc_US_pca_fs"
cap erase "`table'.tex"
cap erase "`table'.txt"

forvalues i=1/7 {
	
	
		ivreg2 ir_THREEFY8 `mp`i'' $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format_fs`i'' dec(4) `text`i''
	

}
cap erase "`table'.txt"



//// APPENDIX - First stage table -  With all controls reported 

local table = "$path/tables_paper/MaSc_US_pca_fs_withcontrols"
cap erase "`table'.tex"
cap erase "`table'.txt"

forvalues i=1/7 {
	
		ivreg2 ir_THREEFY8 `mp`i'' $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format_fs`i'_wc' dec(4) `text`i'_wc'
	
}
cap erase "`table'.txt"



///// MAIN TEXT - Correlation table with CBI shocks

estpost correlate std_cbi_median std_ns std_p1_jkbrwksx std_p1_jkbrwksxns, matrix listwise
esttab using "$path/tables_paper/MaSc_US_pca_correlation.tex", unstack not noobs  replace label nostar nonumbers b(2)



////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
////  Second stage - PCA: All - Controls: All - This is selected from the first stage based on the criterion that Effective F-stat is higher than the critical value. 
////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////

local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

local format_wc = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8)"

local text_ss1 = "addtext(Controls,  \bf X and W )"

local table = "$path/tables_paper/MaSc_US_pca_2sls"
cap erase "`table'.tex"
cap erase "`table'.txt"

/// MAIN TEXT - Second stage table 

foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8 {

ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

}

cap erase "`table'.txt"



//// APPENDIX - Second stage table -  With all controls

local table = "$path/tables_paper/MaSc_US_pca_2sls_withcontrols"
cap erase "`table'.tex"
cap erase "`table'.txt"


foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8 ln_SLG_exval8  {

ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format_wc' dec(2) 

}

cap erase "`table'.txt"



////////////////////////////////////////////////////
////////////////////////////////////////////////////
// MAIN TEXT - Second stage with with controls and Neg and Pos CB share change interactions
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	

preserve	
foreach x of varlist HH row_fo row_fp Monetaryauthority MMFs SLG Banks IC PF IF Other {
gen shr_`x'=(`x'/Total_new)*100
}

tsset yq
gen neg_cb=d.shr_Monetaryauthority<0


*gen qt=(inrange(yq,tq(2017q4),tq(2019q2)) | yq>tq(2022q2)) & !missing(yq) // This one is the baseline
*gen qt=(inrange(yq,tq(2015q1),tq(2019q2)) | yq>tq(2022q2)) & !missing(yq)
*gen qt=(inrange(yq,tq(2017q4),tq(2019q3)) | yq>tq(2022q1)) & !missing(yq)
*gen qt=(inrange(yq,tq(2017q4),tq(2019q3)) | yq>tq(2022q2)) & !missing(yq)
gen qt=neg_cb

gen non_qt=1-qt

gen int_end_qt=ir_THREEFY8 * qt
gen int_iv_qt=std_p1_jkbrwksx * qt

gen int_end_nonqt=ir_THREEFY8 * non_qt
gen int_iv_nonqt=std_p1_jkbrwksx * non_qt

gen post_gfc=yq>tq(2008q4)
label var post_gfc "Post-GFC"


label var qt "$\Delta~CB~Share<0$"
label var int_end_qt "$8Y~Yield~(ZC) \times \Delta~CB~Share<0$"
label var int_end_nonqt "$8Y~Yield~(ZC) \times \Delta~CB~Share\geq0$"

local format_qt_int = "label nor2  addstat(Equal coefficients (p-val), r(p), Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons keep(int_end_nonqt int_end_qt qt) sortvar(int_end_nonqt int_end_qt qt)"

local text_ss1 = "addtext(Controls,  \bf X and W )"

// For the main text
local table = "$path/tables_paper/MaSc_US_negcb_int_2sls"
cap erase "`table'.tex"
cap erase "`table'.txt"

*foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_HH_exval8 ln_SLG_exval8 ln_Other_exval8  {

foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8  {

ivreg2 `x' (int_end_nonqt int_end_qt = int_iv_nonqt int_iv_qt) qt  $ln_controls , robust bw(auto) ffirst
test int_end_nonqt = int_end_qt

outreg2 using "`table'.tex", tex(frag) append `format_qt_int' dec(2) `text_ss1'

}

cap erase "`table'.txt"



// For the appendix - post-gfc dummy
local table = "$path/tables_paper/MaSc_US_negcb_int_2sls_robgfcdum"
cap erase "`table'.tex"
cap erase "`table'.txt"

*foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_HH_exval8 ln_SLG_exval8 ln_Other_exval8  {

foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8  {

ivreg2 `x' (int_end_nonqt int_end_qt = int_iv_nonqt int_iv_qt) qt post_gfc $ln_controls , robust bw(auto) ffirst
test int_end_nonqt = int_end_qt

outreg2 using "`table'.tex", tex(frag) append `format_qt_int' dec(2) `text_ss1'

}

cap erase "`table'.txt"

// For appendix - Robustness check excluding QE1


local table = "$path/tables_paper/MaSc_US_negcb_int_2sls_excqe1rob"
cap erase "`table'.tex"
cap erase "`table'.txt"

*foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_HH_exval8 ln_SLG_exval8 ln_Other_exval8  {

foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8  {

ivreg2 `x' (int_end_nonqt int_end_qt = int_iv_nonqt int_iv_qt) qt  $ln_controls if !inrange(yq,tq(2009q1), tq(2010q1)), robust bw(auto) ffirst
test int_end_nonqt = int_end_qt

outreg2 using "`table'.tex", tex(frag) append `format_qt_int' dec(2) `text_ss1'

}

cap erase "`table'.txt"



restore



////////////////////////////////////////////////////
////////////////////////////////////////////////////
// MAIN TEXT - Second stage with the outside asset - debt
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 			


local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(3) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "
local format_debt_wc = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(3) nonotes nocons sortvar(ir_THREEFY8)  "
*local format_overid = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp), Hansen J stat (p-val), e(jp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

local text_ss1 = "addtext(Controls, \bf X)"


local table = "$path/tables_paper/MaSc_US_pca_2sls_outsideasset_debt"
cap erase "`table'.tex"
cap erase "`table'.txt"

foreach x of varlist  debtoa_diff_IF  debtoa_diff_Banks debtoa_diff_PF debtoa_diff_IC debtoa_diff_SLG  {

ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ds_controls ,robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

}


// APPENDIX -  With all controls


local table = "$path/tables_paper/MaSc_US_pca_2sls_outsideasset_debt_wc"
cap erase "`table'.tex"
cap erase "`table'.txt"

foreach x of varlist  debtoa_diff_IF  debtoa_diff_Banks  debtoa_diff_PF debtoa_diff_IC debtoa_diff_SLG {

ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ds_controls ,robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format_debt_wc' dec(2) `text_ss1'

}

cap erase "`table'.txt"

// First stage for the outside asset with controls
			

local table = "$path/tables_paper/MaSc_US_pca_fs_outsideasset_wc"
cap erase "`table'.tex"
cap erase "`table'.txt"

forvalues i=1/7 {

	ivreg2 ir_THREEFY8 `mp`i'' $ds_controls , robust bw(auto)  ffirst
	outreg2 using "`table'.tex", tex(frag) append `format_fs`i'_wc_ds' dec(4) `text`i'_wc'
	
}
cap erase "`table'.txt"



////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
////  APPENDIX
////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with many controls.
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) "


		local table = "$path/tables_paper/MaSc_US_pca_2sls_manycontrols"
		cap erase "`table'.tex"
		cap erase "`table'.txt"

			
		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8 ln_SLG_exval8 {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls_many ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) 
		}

		cap erase "`table'.txt"



////////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with only linear trend.
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) "

		local table = "$path/tables_paper/MaSc_US_pca_2sls_linear"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8   {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls_linear ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2)  
		}

		cap erase "`table'.txt"
		
		
		
////////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with std_p1_jkbrwksxns
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8)"


		local table = "$path/tables_paper/MaSc_US_pca_2sls_jkmpbrwksxns"
		cap erase "`table'.tex"
		cap erase "`table'.txt"

			
		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8  {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksxns) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2)  
		}

		cap erase "`table'.txt"		
		
		

////////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with residuals of std_p1_jkbrwksx regressed on std_jk_cbi 
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 			
	
	reg std_p1_jkbrwksx std_cbi_median 
	predict resid, residuals
		
	local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY5) keep(ir_THREEFY8)"


		local table = "$path/tables_paper/MaSc_US_pca_2sls_rc_inforesid"
		cap erase "`table'.tex"
		cap erase "`table'.txt"

			
		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8   ln_SLG_exval8  {

		ivreg2 `x' (ir_THREEFY8=resid) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2)  
		}

		cap erase "`table'.txt"		
		
		

////////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with the first two principal components
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format_overid = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp), Hansen J stat (p-val), e(jp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_double_all_2sls"
		cap erase "`table'.tex"
		cap erase "`table'.txt"

		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8  {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx std_p2_jkbrwksx) $ln_controls  ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format_overid' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"
		
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different valuation adjustments - 5 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_valadj5"
		cap erase "`table'.tex"
		cap erase "`table'.txt"

		
		foreach x of varlist ln_row_fo_exval5 ln_row_fp_exval5 ln_IF_exval5  ln_Banks_exval5 ln_PF_exval5 ln_IC_exval5  ln_SLG_exval5 {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"

		
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different valuation adjustments - 10 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_valadj10"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval10 ln_row_fp_exval10 ln_IF_exval10  ln_Banks_exval10  ln_PF_exval10 ln_IC_exval10  ln_SLG_exval10   {

		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"		
		

	
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different yields - 5 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY5) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_5yyield"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8  ln_SLG_exval8   {

		ivreg2 `x' (ir_THREEFY5=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"			
		
		
		
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different yields - 10 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY10) keep(ir_THREEFY10) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_10yyield"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8 ln_SLG_exval8  {
		
		ivreg2 `x' (ir_THREEFY10=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"		
	

		
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different yields and valuation adjustments - 5 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY5) keep(ir_THREEFY5) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_yield_valadj_5y"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval5 ln_row_fp_exval5 ln_IF_exval5 ln_Banks_exval5 ln_PF_exval5 ln_IC_exval5 ln_SLG_exval5  {
		
		ivreg2 `x' (ir_THREEFY5=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"		
	
		
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with different yields and valuation adjustments - 10 years
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY10) keep(ir_THREEFY10) "

		local text_ss1 = "addtext(Controls, \bf X and W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_yield_valadj_10y"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval10 ln_row_fp_exval10 ln_IF_exval10 ln_Banks_exval10 ln_PF_exval10 ln_IC_exval10 ln_SLG_exval10  {
		
		ivreg2 `x' (ir_THREEFY10=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"		
	
///////////////////////////////////////////////////
////////////////////////////////////////////////////
// RC: Second stage with lagged controls
////////////////////////////////////////////////////
//////////////////////////////////////////////////// 
	
	
		local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "

		local text_ss1 = "addtext(Controls, \bf Lagged X and Lagged W)"

		local table = "$path/tables_paper/MaSc_US_pca_2sls_lagcontrols"
		cap erase "`table'.tex"
		cap erase "`table'.txt"


		foreach x of varlist ln_row_fo_exval8 ln_row_fp_exval8 ln_IF_exval8 ln_Banks_exval8 ln_PF_exval8 ln_IC_exval8 ln_SLG_exval8  {
		
		ivreg2 `x' (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls_lag ,robust bw(auto)  ffirst
		outreg2 using "`table'.tex", tex(frag) append `format' dec(2) `text_ss1'

		}

		cap erase "`table'.txt"		
	