
***********************************************************************
* Project: The Demand for Government Debt – Japan               *
* Description:                                                        *
* This script loads and analyzes quarterly data on Japanese          *
* government bond (JGB) holdings by sector. It generates time-series *
* graphs of holding volumes and shares, and runs regressions to      *
* estimate marginal buyers of government debt across policy regimes. *
***********************************************************************
clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/

//////////////////////////////////////////
//////// Japan  //////////////////////////
//////////////////////////////////////////

use "$path/generated_data/JP_data_v202507.dta", clear
keep if !missing(Total)
keep if inrange(yq,tq(2001q1),tq(2024q2))

//////////////////////////////////////////
//////////////////////////////////////////
//    Graphs
//////////////////////////////////////////
//////////////////////////////////////////

	//////////////////////////////////////////
	// Volumes
	//////////////////////////////////////////
	
	
gen a_1=Centralbank
gen a_2= a_1 + Banks
gen a_3=a_2 + ICPF
gen a_4=a_3 + ROW
gen a_5=a_4 + Publicpensions
gen a_6=a_5 + HH
gen a_7=a_6 + Others

label var a_1 "Central bank"
label var a_2 "Banks"
label var a_3 "ICPF"
label var a_4 "ROW"
label var a_5 "Public Pensions" 
label var a_6 "HH"
label var a_7 "Others"
label var debt_gdp_jp "Debt/GDP (per cent)"

cap gen Covid=1100
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}

/*
cap gen GFC=1100
foreach i of numlist `=tq(2008q3)' {
	cap gen y`i'=`i'
}
*/
cap gen QQE=1100
foreach i of numlist `=tq(2013q1)' {
	cap gen y`i'=`i'
}


local y=tq(2001q1)
twoway   (area a_7 yq, bcolor(black*0.1)) (area a_6 yq, bcolor(black*0.2)) (area a_5 yq, bcolor(black*0.3)) (area a_4 yq, bcolor(black*0.4)) ///
  (area a_3 yq, bcolor(black*0.6)) (area a_2 yq, bcolor(black*0.8)) ///
  (area a_1 yq, bcolor(black)) /// (line debt_gdp_jp yq, yaxis(2) lcolor(black) lpattern(dash) lwidth(medthick)) ///
  (dropline Covid y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  (dropline QQE y`=tq(2013q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  if year>=`y', ytitle("¥ trillions") xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
  legend(rows(2) order(7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(5) size(*.9)  symxsize(*.4))  ///
  text(1150 `=tq(2020q1)' "Covid-19") text(1150 `=tq(2013q1)' "QQE") ylabel(0(200)1100)
graph export "$path/graphs_paper/JP_volumes.pdf",replace



	//////////////////////////////////////////
	// Shares
	//////////////////////////////////////////
	

gen b_1=shr_Centralbank
gen b_2=b_1 + shr_Banks
gen b_3=b_2 + shr_ICPF
gen b_4=b_3 + shr_ROW
gen b_5=b_4 + shr_Publicpensions
gen b_6=b_5 + shr_HH
gen b_7=b_6 + shr_Others

label var b_1 "Central bank"
label var b_2 "Banks"
label var b_3 "ICPF"
label var b_4 "ROW"
label var b_5 "Public Pensions" 
label var b_6 "HH"
label var b_7 "Others"

cap gen Covid_2=100
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}
/*
cap gen GFC_2=100
foreach i of numlist `=tq(2008q3)' {
	cap gen y`i'=`i'
}
*/
cap gen QQE_2=100
foreach i of numlist `=tq(2013q1)' {
	cap gen y`i'=`i'
}

local y=tq(2001q1)
twoway   (area b_7 yq, bcolor(black*0.1)) (area b_6 yq, bcolor(black*0.2)) (area b_5 yq, bcolor(black*0.3)) (area b_4 yq, bcolor(black*0.4)) ///
  (area b_3 yq, bcolor(black*0.6)) (area b_2 yq, bcolor(black*0.8)) ///
  (area b_1 yq, bcolor(black)) (dropline Covid_2 y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  (dropline QQE_2 y`=tq(2013q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
  if year>=`y', ytitle("Per cent") xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
  legend(rows(2) order(7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(5) size(*.9)  symxsize(*.4))  ///
  text(103 `=tq(2020q1)' "Covid-19") text(103 `=tq(2013q1)' "QQE")
graph export "$path/graphs_paper/JP_shares.pdf",replace



	//////////////////////////////////////////
	// Regressions
	//////////////////////////////////////////
	
foreach x of varlist Centralbank-Others Total {
	
	gen d`x'=d.`x'
}

foreach x of varlist Centralbank-Others Total {
	
		gen d`x'_n = d`x'/l.Total
}

gen tot_preqqe=dTotal_n*pre_qqe
gen tot_postqqe=dTotal_n*post_qqe
gen tot_pc=dTotal_n*post_covid


**** Marginal holders of JGBs
label var dCentralbank_n "CB"
label var dROW_n "ROW"
label var dICPF_n "ICPF"
label var dBanks_n "Banks"
label var dHH_n "HH"
label var dPublicpensions_n "PP"
label var dOthers_n "Others"

label var tot_preqqe "Pre-QQE * Pct. Ch. Gov. Debt"
label var tot_postqqe "QQE * Pct. Ch. Gov. Debt"
label var tot_pc "Post-Covid * Pct. Ch. Gov. Debt"


local format = "nocons nonotes dec(2) label"

local reg_table = "$path/tables_paper/JP_marginal_holders"
cap erase "`reg_table'.tex"
cap erase "`reg_table'.txt"

ivreg2 dCentralbank_n tot_preqqe tot_postqqe tot_pc ,  bw(auto)

outreg2 using "`reg_table'.tex", tex(frag) replace `format' 

foreach x of varlist dBanks_n-dOthers_n  {
	
ivreg2 `x' tot_preqqe tot_postqqe tot_pc , robust bw(auto)
outreg2 using "`reg_table'.tex", tex(frag) append `format'

}

cap erase "`reg_table'.txt"


/// averages
preserve
gen y=.
replace y=1 if pre_qqe
replace y=2 if post_qqe 
replace y=3 if post_covid
collapse shr_*,by(y)

rename y shr_y

foreach var of varlist _all {

replace `var'=round(`var',0.1)

}

rename shr_* *
rename Centralbank CB
rename Publicpensions PP

label var y "Avg. Share"
label define time 1 "Pre-QQE" 2 "QQE" 3 "Post-Covid"
label values y time

order y CB Banks ICPF ROW PP HH Others

estpost tabstat CB Banks ICPF ROW PP HH Others, by(y) nototal
esttab using  "$path/tables_paper/JP_avg_by_time.tex", cells("CB Banks ICPF ROW PP HH Others") noobs nomtitle nonumber varwidth(30) collab(, lhs("`:var lab y'")) tex replace
restore

