***********************************************************************
* Project: The Demand for Government Debt – United States             *
* Description:                                                        *
* This script constructs a quarterly dataset on U.S. Treasury         *
* holdings by sector, merged with macro-financial indicators and      *
* various monetary policy shock series. It supports regression-based  *
* analysis of sovereign debt demand and the changing composition of   *
* bondholders across policy regimes.                                  *
***********************************************************************

clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/

//////////////////////////////////////////
//////// United States   /////////////////
//////////////////////////////////////////

use "$path/generated_data/US_data_v202507.dta", clear
drop if HH==.

keep if inrange(yq,tq(2001q1),tq(2024q2))

//////////////////////////////////////////
//////////////////////////////////////////
//    Graphs
//////////////////////////////////////////
//////////////////////////////////////////

	//////////////////////////////////////////
	// Volumes
	//////////////////////////////////////////

gen a_1 = Monetaryauthority 
gen a_2 = a_1 + row_fo
gen a_3 = a_2 + row_fp
gen a_4 = a_3 + IF
gen a_5 = a_4 + Banks
gen a_6 = a_5 + PF
gen a_7 = a_6 + IC
gen a_8 = a_7 + HH
gen a_9 = a_8 + SLG
gen a_10 = a_9 + MMFs
gen a_11 = a_10 + Other

label var a_1 "Central bank"
label var a_2 "FOff"
label var a_3 "FPri"
label var a_4 "IF"
label var a_5 "Banks"
label var a_6 "PF"
label var a_7 "IC"
label var a_8 "HH"
label var a_9 "SLG"
label var a_10 "MMF"
label var a_11 "Other"
label var debt_gdp_us "Debt/GDP (per cent)"

cap gen Covid=25
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}

cap gen QT1=25
foreach i of numlist `=tq(2017q4)' {
	cap gen y`i'=`i'
}

cap gen QE=25
foreach i of numlist `=tq(2009q1)' {
	cap gen y`i'=`i'
}

cap gen QT2=25
foreach i of numlist `=tq(2022q2)' {
	cap gen y`i'=`i'
}

local y=tq(2001q1)
twoway (area a_11 yq, bcolor(black*0.1)) (area a_10 yq, bcolor(black*0.15)) (area a_9 yq, bcolor(black*0.2)) ///
(area a_8 yq, bcolor(black*0.3)) (area a_7 yq, bcolor(black*0.4)) (area a_6 yq, bcolor(black*0.5))  ///
(area a_5 yq, bcolor(black*0.6)) (area a_4 yq, bcolor(black*0.7))  ///
(area a_3 yq, bcolor(black*0.8))  (area a_2 yq, bcolor(black*0.9))  (area a_1 yq, bcolor(black)) /// (line debt_gdp_us yq, yaxis(2) lcolor(black) lpattern(dash) lwidth(medthick)) ///
(dropline QE y`=tq(2009q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline Covid y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline QT1 y`=tq(2017q4)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline QT2 y`=tq(2022q2)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
if yq>=`y', ytitle("$ trillions") xtitle(" ") graphre(color(white))  ylabel(,nogrid)  ///
legend(rows(2) order(11 10 9 8 7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(6) size(*.9)  symxsize(*.4)) ///
text(26.5 `=tq(2020q1)' "Covid-19") text(25.5 `=tq(2009q1)' "QE") text(25.5 `=tq(2017q4)' "QT1") text(25.5 `=tq(2022q2)' "QT2")
graph export "$path/graphs_paper/US_volumes.pdf",replace

	//////////////////////////////////////////
	// Shares
	//////////////////////////////////////////	
	
foreach x of varlist HH row_fo row_fp Monetaryauthority MMFs SLG Banks IC PF IF Other {
gen shr_`x'=(`x'/Total_new)*100
}

gen b_1=shr_Monetaryauthority 
gen b_2=b_1 + shr_row_fo
gen b_3=b_2 + shr_row_fp
gen b_4=b_3 + shr_IF
gen b_5=b_4 + shr_Banks
gen b_6=b_5 + shr_PF
gen b_7=b_6 + shr_IC
gen b_8=b_7 + shr_HH
gen b_9=b_8 + shr_SLG
gen b_10=b_9 + shr_MMFs
gen b_11=b_10 + shr_Other

label var b_1 "Central bank"
label var b_2 "FOff"
label var b_3 "FPri"
label var b_4 "IF"
label var b_5 "Banks"
label var b_6 "PF"
label var b_7 "IC"
label var b_8 "HH"
label var b_9 "SLG"
label var b_10 "MMF"
label var b_11 "Other"

cap gen Covid_2=100
foreach i of numlist `=tq(2020q1)' {
	cap gen y`i'=`i'
}

cap gen QT1_2=100
foreach i of numlist `=tq(2017q4)' {
	cap gen y`i'=`i'
}

cap gen QE_2=100
foreach i of numlist `=tq(2009q1)' {
	cap gen y`i'=`i'
}

cap gen QT2_2=100
foreach i of numlist `=tq(2022q2)' {
	cap gen y`i'=`i'
}


local y=tq(2001q1)
twoway  (area b_11 yq, bcolor(black*0.1)) (area b_10 yq, bcolor(black*0.15)) (area b_9 yq, bcolor(black*0.2)) (area b_8 yq, bcolor(black*0.3)) (area b_7 yq, bcolor(black*0.4)) ///
(area b_6 yq, bcolor(black*0.5))  (area b_5 yq, bcolor(black*0.6)) (area b_4 yq, bcolor(black*0.7))  ///
(area b_3 yq, bcolor(black*0.8)) (area b_2 yq, bcolor(black*0.9))  (area b_1 yq, bcolor(black)) ///
(dropline QE_2 y`=tq(2009q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline Covid_2 y`=tq(2020q1)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline QT1_2 y`=tq(2017q4)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
(dropline QT2_2 y`=tq(2022q2)', lcolor(red) mcolor(none) lwidth(thin) lpattern(dash_dot)) ///
if yq>=`y', ytitle("Per cent") xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
legend(rows(2) order(11 10 9 8 7 6 5 4 3 2 1) region(lwidth(none)) pos(bottom) col(6) size(*.9)  symxsize(*.4))  ///
text(105 `=tq(2020q1)' "Covid-19") text(105 `=tq(2009q1)' "QE") text(105 `=tq(2017q4)' "QT1") text(105 `=tq(2022q2)' "QT2")
graph export "$path/graphs_paper/US_shares.pdf",replace



//////////////////////////////////////////
//////////////////////////////////////////
//    Regressions
//////////////////////////////////////////
//////////////////////////////////////////
foreach x of varlist HH Monetaryauthority MMFs Banks IC PF IF SLG Other row_fp row_fo Total_new {
	gen d`x'=d.`x'
}

foreach x of varlist HH Monetaryauthority MMFs Banks IC PF IF SLG Other row_fp row_fo Total_new {
	gen d`x'_n = d`x'/l.Total_new
}


gen pre_gfc=yq<tq(2009q1) & !missing(yq)
gen qt1=inrange(yq,tq(2017q4),tq(2019q4))
*drop post_gfc
gen post_gfc=inrange(yq,tq(2009q1),tq(2017q3))
gen qt2=yq>tq(2022q2) & !missing(yq)
*drop post_covid
gen post_covid=inrange(yq,tq(2020q1),tq(2022q1))


gen tot_beforegfc=dTotal_new_n*pre_gfc
gen tot_pgfc=dTotal_new_n*post_gfc
gen tot_qt1=dTotal_new_n*qt1
gen tot_pc=dTotal_new_n*post_covid
gen tot_qt2=dTotal_new_n*qt2

**** Marginal holders of US Treasuries


label var dMonetaryauthority_n "CB"
*label var dROW_n "ROW"
label var drow_fp_n "FPri"
label var drow_fo_n "FOff"
label var dIC_n "IC"
label var dBanks_n "Banks"
label var dPF_n "PF"
label var dIF_n "IF"
label var dSLG_n "SLG"
label var dMMFs_n "MMF"
label var dHH_n "HH"
label var dOther_n "Other"

label var tot_beforegfc "Pre-GFC * Pct. Ch. Gov. Debt"
label var tot_pgfc "Post-GFC * Pct. Ch. Gov. Debt"
label var tot_qt1 "QT1 * Pct. Ch. Gov. Debt"
label var tot_pc "Covid * Pct. Ch. Gov. Debt"
label var tot_qt2 "QT2 * Pct. Ch. Gov. Debt"


local format = "nocons nonotes dec(2) label"

local reg_table = "$path/tables_paper/US_marginal_holders"
cap erase "`reg_table'.tex"
cap erase "`reg_table'.txt"

ivreg2 dMonetaryauthority_n tot_beforegfc tot_pgfc tot_qt1 tot_pc tot_qt2, bw(auto)

outreg2 using "`reg_table'.tex", tex(frag) replace `format' 

foreach x of varlist drow_fo_n drow_fp_n dIF_n dBanks_n  dPF_n dIC_n dHH_n  dSLG_n dMMFs_n dOther_n {
	
ivreg2 `x' tot_beforegfc tot_pgfc tot_qt1 tot_pc tot_qt2, robust bw(auto)
outreg2 using "`reg_table'.tex", tex(frag) append `format'

}

cap erase "`reg_table'.txt"



/// averages
preserve
*rename shr_ICPF share_ICPF
gen y=.
replace y=1 if pre_gfc
replace y=2 if post_gfc 
replace y=3 if qt1
replace y=4 if post_covid
replace y=5 if qt2
collapse shr_*,by(y)

rename y shr_y

foreach var of varlist _all {

replace `var'=round(`var',0.1)

}

rename shr_* *
rename Monetaryauthority CB
rename MMFs MMF

label var y "Avg. Share"
label define time 1 "Pre-GFC" 2 "Post-GFC" 3 "QT1" 4 "Covid" 5 "QT2"
label values y time

order y CB row_fo row_fp IF Banks PF IC  HH SLG MMF Other

label var row_fo "FOff"
label var row_fp "FPri"
rename row_fo FOff
rename row_fp FPri
estpost tabstat CB FOff FPri IF Banks PF IC HH  SLG MMF Other, by(y) nototal
esttab using  "$path/tables_paper/US_avg_by_time.tex", cells("CB FOff FPri PF IF Banks SLG MMF HH IC Other") noobs nomtitle nonumber varwidth(30) collab(, lhs("`:var lab y'")) tex replace label
restore
