***********************************************************************
* Project: The Demand for Government Debt – United States            *
* Description:                                                       *
* This script simulates counterfactual scenarios to quantify the     *
* impact of different investor sectors on the price elasticity of    *
* demand for U.S. government debt. It estimates sectoral elasticities*
* and simulates yield impacts under various exclusion scenarios      *
* (e.g., without foreign official holders, banks, etc.),             *
* especially during QE and QT episodes.                              *
***********************************************************************
clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/



use "$path/generated_data/us_regression_cf_v202507.dta",clear

/// Regression

global ln_controls "ln_GDP dln_GDP ir_inf_exc_fe ln_VIXCLS ir_german_5y_zc ln_NBUS yq yq_sq" 


tsset yq

tempfile results_qt
local replace replace
foreach lhs in "ln_row_fo_exval8" "ln_row_fp_exval8" "ln_IF_exval8" "ln_Banks_exval8" "ln_PF_exval8" "ln_IC_exval8" "ln_SLG_exval8"  {
	
		xi: ivreg2 `lhs' (ir_THREEFY8=std_p1_jkbrwksx)  $ln_controls , robust bw(auto)
		regsave using "`results_qt'", pval autoid `replace' addlabel(lhs,"`lhs'") 
		local replace append
	
}

use "`results_qt'", clear
list

gen sector=substr(lhs,4,length(lhs)-10)
keep if inlist(var,"ir_THREEFY8") 
*replace var=substr(var,9,.)
preserve
use "$path/generated_data/US_data_v202507.dta",clear
egen max_yq=max(yq) if !missing(Banks)
keep if yq==max_yq
keep Banks PF IC row_fo row_fp IF SLG HH
xpose, clear v
rename _varname sector
rename v1 latest_holdings
drop v2
tempfile hold
save `hold'
restore

merge m:1 sector using `hold',nogen

sort sector var

gen coef_95_low=coef-1.96*stderr 
gen coef_95_up=coef+1.96*stderr 

gen coef_90_low=coef-1.64*stderr 
gen coef_90_up=coef+1.64*stderr 


set obs 10
replace sector="Monetaryauthority" if _n==9
replace sector="Other" if _n==10

replace coef=0 if inlist(sector,"HH","Monetaryauthority","Other")

keep sector coef

preserve
use "$path/generated_data/US_data_v202507.dta", clear
drop if HH==.

keep if inrange(yq,tq(2001q1),tq(2024q2))

keep HH row_fo row_fp Monetaryauthority MMFs SLG Banks IC PF IF Other yq

ds yq,not
foreach i in `r(varlist)' {
	rename `i' hold`i'
}

reshape long hold,i(yq) j(sector,s)


tempfile holdings
save `holdings'
restore

merge 1:m sector using `holdings',nogen

*rop if sector=="Other"
drop if sector=="MMFs"

encode sector,gen(sector_enc)

xtset sector_enc yq

bys yq: egen Total_actual=sum(hold)
gen shr_act=hold/Total_actual
gen wgt_coef=coef*shr_act
bys yq: egen mkt_elas_yq_actual=sum(wgt_coef)

/// Get the portfolio weights to calculate the price elasticity a la Koijen and Yogo (2019)

preserve
use "$path/generated_data/US_data_v202507.dta", clear

*gen w_exUST_HH=1-(HH/HH_debt)
gen w_exUST_SLG=1-(SLG/SLG_debt)
gen w_exUST_Banks=1-(Banks/Banks_debt)
gen w_exUST_PF=1-(PF/PF_debt)
gen w_exUST_IF=1-(IF/IF_debt)
gen w_exUST_IC=1-(IC/IC_debt)
gen w_exUST_row_fp=1-(row_fp/(ROW_debt*pri_shr))
gen w_exUST_row_fo=(100-dollar_share)/100

keep yq w_exUST*

reshape long w_exUST_,i(yq) j(sector,s)

tempfile gh
save `gh'
restore

merge 1:1 yq sector using `gh',nogen
drop if missing(coef)


replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_act=1+(coef/8)*w_exUST_

gen wgt_elas_act=shr_act*ind_elas_act
bys yq: egen mkt_pri_elas_yq_act=sum(wgt_elas_act)


// Baseline

preserve
drop if sector=="Monetaryauthority"

bys yq: egen Total_base=sum(hold)
gen shr_base=hold/Total_base
gen wgt_coef_base=coef*shr_base
bys yq: egen mkt_elas_yq_base=sum(wgt_coef_base)

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_base=1+(coef/8)*w_exUST_

gen wgt_elas_base=shr_base*ind_elas_base
bys yq: egen mkt_pri_elas_yq_base=sum(wgt_elas_base)

collapse mkt_elas_yq_base mkt_pri_elas_yq_base,by(yq)

tempfile base
save `base'
restore

merge m:1 yq using `base',nogen

/*
// No Fed
preserve
drop if sector=="Monetaryauthority"

bys yq: egen Total_nofed=sum(hold)
gen shr_nofed=hold/Total_nofed
gen wgt_coef_nofed=coef*shr_nofed
bys yq: egen mkt_elas_yq_nofed=sum(wgt_coef_nofed)

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_nofed=1+(coef/8)*w_exUST_

gen wgt_elas_nofed=shr_nofed*ind_elas_nofed
bys yq: egen mkt_pri_elas_yq_nofed=sum(wgt_elas_nofed)

collapse mkt_elas_yq_nofed mkt_pri_elas_yq_nofed,by(yq)

tempfile nofed
save `nofed'
restore

merge m:1 yq using `nofed',nogen
*/

// No Foreign official
preserve
drop if sector=="row_fo"
drop if sector=="Monetaryauthority"

bys yq: egen Total_nofo=sum(hold)
gen shr_nofo=hold/Total_nofo
gen wgt_coef_nofo=coef*shr_nofo
bys yq: egen mkt_elas_yq_nofo=sum(wgt_coef_nofo)

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_nofo=1+(coef/8)*w_exUST_


gen wgt_elas_nofo=shr_nofo*ind_elas_nofo
bys yq: egen mkt_pri_elas_yq_nofo=sum(wgt_elas_nofo)

collapse mkt_pri_elas_yq_nofo mkt_elas_yq_nofo,by(yq)

tempfile nofo
save `nofo'
restore

merge m:1 yq using `nofo',nogen


// No Foreign private
preserve
drop if sector=="row_fp"
drop if sector=="Monetaryauthority"

bys yq: egen Total_nofp=sum(hold)
gen shr_nofp=hold/Total_nofp
gen wgt_coef_nofp=coef*shr_nofp
bys yq: egen mkt_elas_yq_nofp=sum(wgt_coef_nofp)

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_nofp=1+(coef/8)*w_exUST_


gen wgt_elas_nofp=shr_nofp*ind_elas_nofp
bys yq: egen mkt_pri_elas_yq_nofp=sum(wgt_elas_nofp)

collapse mkt_elas_yq_nofp mkt_pri_elas_yq_nofp,by(yq)

tempfile nofp
save `nofp'
restore

merge m:1 yq using `nofp',nogen

// No Foreign private/official
preserve
drop if sector=="row_fp"
drop if sector=="row_fo"
drop if sector=="Monetaryauthority"

bys yq: egen Total_nofpfo=sum(hold)
gen shr_nofpfo=hold/Total_nofpfo
gen wgt_coef_nofpfo=coef*shr_nofpfo
bys yq: egen mkt_elas_yq_nofpfo=sum(wgt_coef_nofpfo)

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_nofpfo=1+(coef/8)*w_exUST_

gen wgt_elas_nofpfo=shr_nofpfo*ind_elas_nofpfo
bys yq: egen mkt_pri_elas_yq_nofpfo=sum(wgt_elas_nofpfo)

collapse mkt_pri_elas_yq_nofpfo mkt_elas_yq_nofpfo,by(yq)

tempfile nofpfo
save `nofpfo'
restore

merge m:1 yq using `nofpfo',nogen



foreach i in Banks IF PF IC SLG HH {
preserve
drop if sector=="`i'"
drop if sector=="Monetaryauthority"

bys yq: egen Total_no`i'=sum(hold)
gen shr_no`i'=hold/Total_no`i'
gen wgt_coef_no`i'=coef*shr_no`i'
bys yq: egen mkt_elas_yq_no`i'=sum(wgt_coef_no`i')

replace w_exUST_=1 if missing(w_exUST_)
gen ind_elas_no`i'=1+(coef/8)*w_exUST_


gen wgt_elas_no`i'=shr_no`i'*ind_elas_no`i'
bys yq: egen mkt_pri_elas_yq_no`i'=sum(wgt_elas_no`i')

collapse mkt_pri_elas_yq_no`i' mkt_elas_yq_no`i',by(yq)

tempfile no`i'
save `no`i''
restore

merge m:1 yq using `no`i'',nogen
}


gen impact_actual=(1/mkt_elas_yq_actual)*100
*gen impact_nofed=(1/mkt_elas_yq_nofed)
gen impact_nofo=(1/mkt_elas_yq_nofo)*100
*gen impact_nofofed=(1/mkt_elas_yq_nofofed)
gen impact_base=(1/mkt_elas_yq_base)*100
gen impact_nofp=(1/mkt_elas_yq_nofp)*100
gen impact_nofpfo=(1/mkt_elas_yq_nofpfo)*100
gen impact_noBanks=(1/mkt_elas_yq_noBanks)*100 
gen impact_noIF=(1/mkt_elas_yq_noIF)*100 
gen impact_noPF=(1/mkt_elas_yq_noPF)*100 
gen impact_noIC=(1/mkt_elas_yq_noIC)*100 
gen impact_noHH=(1/mkt_elas_yq_noHH)*100 
gen impact_noSLG=(1/mkt_elas_yq_noSLG)*100 

label var impact_base "Baseline"
label var impact_nofo "w/o Foreign Official"
label var impact_nofp "w/o Foreign Private"
label var impact_nofpfo "w/o Foreigners"
label var impact_noBanks "w/o Banks"
label var impact_noIF "w/o IF"
label var impact_noPF "w/o PF"
label var impact_noIC "w/o IC"
label var impact_noHH "w/o HH"
label var impact_noSLG "w/o SLG"


* Create a graph with shaded areas and the time series

tw (line impact_base yq,lwidth(vthick) lcolor(black)) (line impact_nofo yq,lcolor(blue)) (line impact_nofp yq,lcolor(midgreen)) (line impact_nofpfo yq,lcolor(gray))  (line impact_noBanks yq,lcolor(cranberry)) (line impact_noIF yq,lcolor(purple)) (line impact_noPF yq,lcolor(cyan)) (line impact_noIC yq,lcolor(yellow))  (line impact_noSLG yq,lcolor(maroon)) (line impact_noHH yq,lcolor(orange)), scheme(stcolor) legend(pos(6) rows(2) symysize(*0.5) size(small)) xtitle(" ") ytitle("Impact on yields of 1% higher residual supply (bps)")
graph export "$path/graphs_paper/timeseries_yield_counterfactuals.pdf",replace 

label var mkt_pri_elas_yq_base "Baseline"
label var mkt_pri_elas_yq_nofo "w/o Foreign Official"
label var mkt_pri_elas_yq_nofp "w/o Foreign Private"
label var mkt_pri_elas_yq_nofpfo "w/o Foreigners"
label var mkt_pri_elas_yq_noBanks "w/o Banks"
label var mkt_pri_elas_yq_noIF "w/o IF"
label var mkt_pri_elas_yq_noPF "w/o PF"
label var mkt_pri_elas_yq_noIC "w/o IC"
label var mkt_pri_elas_yq_noHH "w/o HH"
label var mkt_pri_elas_yq_noSLG "w/o SLG"


tw (line mkt_pri_elas_yq_base yq,lwidth(vthick) lcolor(black)) (line mkt_pri_elas_yq_nofo yq,lcolor(blue)) (line mkt_pri_elas_yq_nofp yq,lcolor(midgreen)) (line mkt_pri_elas_yq_nofpfo yq,lcolor(gray))  (line mkt_pri_elas_yq_noBanks yq,lcolor(cranberry)) (line mkt_pri_elas_yq_noIF yq,lcolor(purple)) (line mkt_pri_elas_yq_noPF yq,lcolor(cyan)) (line mkt_pri_elas_yq_noIC yq,lcolor(yellow))  (line mkt_pri_elas_yq_noSLG yq,lcolor(maroon)) (line mkt_pri_elas_yq_noHH yq,lcolor(orange)), scheme(stcolor) legend(pos(6) rows(2) symysize(*0.5) size(small)) xtitle(" ") ytitle("Weighted average price elasticity of demand of select sectors", size(small))
graph export "$path/graphs_paper/timeseries_prielas_counterfactuals.pdf",replace 



// QE & QT impact

gen qe1=inrange(yq,tq(2009q1),tq(2010q1))
gen qe2=inrange(yq,tq(2010q3),tq(2011q2))
gen qe3=inrange(yq,tq(2012q4),tq(2014q4))
gen qe4=inrange(yq,tq(2019q4),tq(2021q4))
gen qt1=inrange(yq,tq(2017q4),tq(2019q3))
gen qt2=inrange(yq,tq(2022q1),tq(2024q2))

gen period="QE1" if qe1
replace period="QE2" if qe2
replace period="QE3" if qe3
replace period="QE4" if qe4
replace period="QT1" if qt1
replace period="QT2" if qt2
replace period="Other" if missing(period)

preserve
bys period: egen min_date=min(yq)
bys period: egen max_date=max(yq)
sort yq
format min_date %tq
format max_date %tq
drop if period =="Other"
keep if yq==min_date | yq==max_date
sort period yq

keep if sector=="Monetaryauthority"
bys period: gen diff_cb=(shr_act[_n]-shr_act[_n-1])*100
bys period: replace diff_cb=diff_cb[_n+1]

drop if diff_cb==.
gen price_impact_base=-diff_cb*impact_base
gen price_impact_nofo=-diff_cb*impact_nofo
gen price_impact_nofp=-diff_cb*impact_nofp
gen price_impact_noBanks=-diff_cb*impact_noBanks
gen price_impact_noIF=-diff_cb*impact_noIF
gen price_impact_noPF=-diff_cb*impact_noPF
gen price_impact_noIC=-diff_cb*impact_noIC
gen price_impact_noHH=-diff_cb*impact_noHH
gen price_impact_noSLG=-diff_cb*impact_noSLG
gen price_impact_fpfono=-diff_cb*impact_nofpfo

gen price_impact_qeqtcount=-diff_cb*8.94 if substr(period,1,2)=="QE" // 8.94 is the baseline price impact observed at the last quarter
replace price_impact_qeqtcount=-diff_cb*13.7 if substr(period,1,2)=="QT"  // 13.7 is the maximum baseline price impact throughout the sample

label var price_impact_base "Baseline"
label var price_impact_qeqtcount "QE/QT counterfactual"
*label var price_impact_qtthen "QT counterfactual"

foreach v in fo fp Banks IF PF IC HH SLG {
	gen price_impact_diff_`v'= price_impact_no`v' - price_impact_base 
}

gen price_impact_diff_fpfo=price_impact_fpfono-price_impact_base

graph bar price_impact_base price_impact_qeqtcount ,over(period) ytitle("Impact on yields (bps)") legend(pos(6) rows(1) label(1 "Baseline") label(2 "QE/QT counterfactual")) scheme(stcolor) bar(1, color(black))  bar(2, color(gray))
graph export "$path/graphs_paper/qe_qt_effect_diffelas.pdf",replace


graph bar price_impact_base price_impact_nofo price_impact_nofp price_impact_fpfono,over(period) ytitle("Impact on yields (bps)") legend(pos(6) rows(1) label(1 "Baseline") label(2 "Without Foreign Official") label(3 "Without Foreign Private") label(4 "Without Foreigners") symysize(*0.5) size(small))  scheme(stcolor)  bar(1, color(black)) bar(1, color(black)) bar(2, color(blue)) bar(3, color(midgreen)) bar(4, color(gray))
graph export "$path/graphs_paper/qe_qt_effect_fofpcounter.pdf",replace


graph bar price_impact_base price_impact_nofo price_impact_nofp price_impact_fpfono price_impact_noBanks price_impact_noIF price_impact_noPF price_impact_noIC price_impact_noHH price_impact_noSLG,over(period) ytitle("Impact on yields (bps)") legend(pos(6) rows(2) label(1 "Baseline") label(2 "w/o Foreign Official") label(3 "w/o Foreign Private") label(4 "w/o Foreigners") label(5 "w/o Banks")  label(6 "w/o IF") label(7 "w/o PF") label(8 "w/o IC") label(9 "w/o HH") label(10 "w/o SLG") symysize(*0.5) size(small))  scheme(stcolor) ylabel(-150(25)100)  bar(1, color(black)) bar(2, color(blue)) bar(3, color(midgreen)) bar(4, color(gray)) bar(5, color(cranberry)) bar(6, color(purple)) bar(7, color(cyan)) bar(8,color(yellow)) bar(9, color(maroon)) bar(10, color(orange))
graph export "$path/graphs_paper/qe_qt_effect_allcounter.pdf",replace


graph bar price_impact_base price_impact_nofo price_impact_nofp price_impact_fpfono price_impact_noBanks price_impact_noIF,over(period) ytitle("Impact on yields (bps)") legend(pos(6) rows(2) label(1 "Baseline") label(2 "w/o Foreign Official") label(3 "w/o Foreign Private") label(4 "w/o Foreigners") label(5 "w/o Banks")  label(6 "w/o IF") symysize(*0.5) size(small))  scheme(stcolor) ylabel(-150(25)100) bar(1, color(black)) bar(1, color(black)) bar(2, color(blue)) bar(3, color(midgreen)) bar(4, color(gray)) bar(5, color(cranberry)) bar(6, color(purple))
graph export "$path/graphs_paper/qe_qt_effect_somecounter.pdf",replace

cap gen price_impact_diff_base=0
graph bar price_impact_diff_base price_impact_diff_fo price_impact_diff_fp price_impact_diff_fpfo price_impact_diff_Banks price_impact_diff_IF price_impact_diff_PF price_impact_diff_IC price_impact_diff_HH price_impact_diff_SLG  ,over(period) ytitle("Difference in LT yields compared to baseline (bps)") legend(pos(6) rows(2) label(1 "Baseline" ) label(2 "w/o Foreign Official") label(3 "w/o Foreign Private") label(4 "w/o Foreigners") label(5 "w/o Banks")  label(6 "w/o IF") label(7 "w/o PF") label(8 "w/o IC") label(9 "w/o HH") label(10 "w/o SLG") symysize(*0.5) size(small))  scheme(stcolor) ylabel(-50(10)50)  bar(1, color(black)) bar(2, color(blue)) bar(3, color(midgreen)) bar(4, color(gray)) bar(5, color(cranberry)) bar(6, color(purple)) bar(7, color(cyan)) bar(8,color(yellow)) bar(9, color(maroon)) bar(10, color(orange))
graph export "$path/graphs_paper/qe_qt_effect_allcounter_diff.pdf",replace

restore


////////////// 
preserve
use "$path/generated_data/us_regression_cf_v202507.dta",clear

egen Total_abs=rowtotal(HH row_fo row_fp SLG Banks IC PF IF)

foreach x of varlist HH row_fo row_fp SLG Banks IC PF IF {
gen shr_`x'=(`x'/Total_abs)*100
}

label var shr_row_fp "ROW (Pri.)"
label var shr_row_fo "ROW (Off.)"
label var shr_HH "HH"
label var shr_Banks "Banks"
label var shr_IF "IF"
label var shr_PF "PF"
label var shr_IC "IC"
label var shr_SLG "SLG"

tw (line shr_row_fo yq,lcolor(blue)) (line shr_row_fp yq,lcolor(midgreen)) (line shr_Banks yq,lcolor(cranberry)) (line shr_IF yq,lcolor(purple)) (line shr_PF yq,lcolor(cyan)) (line shr_IC yq,lcolor(yellow)) (line shr_SLG yq, lcolor(maroon)) (line shr_HH  yq,lcolor(orange)) if inrange(yq, tq(2001q1), tq(2024q2)), xtitle(" ") ytitle("Share among selected sectors") scheme(stcolor) legend(pos(6) rows(1) symysize(*0.5) size(small))
graph export "$path/graphs_paper/share_among_select_sectors.pdf",replace
restore


//// Individual price elasticity
preserve 
keep yq sector ind_elas_act
reshape wide ind_elas_act,i(yq) j(sector,s)

label var ind_elas_actrow_fp "ROW (Pri.)"
label var ind_elas_actrow_fo "ROW (Off.)"
label var ind_elas_actHH "HH"
label var ind_elas_actBanks "Banks"
label var ind_elas_actIF "IF"
label var ind_elas_actPF "PF"
label var ind_elas_actIC "IC"
label var ind_elas_actSLG "SLG"

tw (line ind_elas_actrow_fo yq,lcolor(blue)) (line ind_elas_actrow_fp yq,lcolor(midgreen)) (line ind_elas_actBanks yq,lcolor(cranberry)) (line ind_elas_actIF yq,lcolor(purple)) (line ind_elas_actPF yq,lcolor(cyan)) (line ind_elas_actIC yq,lcolor(yellow)) (line ind_elas_actSLG yq, lcolor(maroon)) (line ind_elas_actHH  yq,lcolor(orange)) if inrange(yq, tq(2001q1), tq(2024q2)), xtitle(" ") ytitle("Price elasticity off selected sectors") scheme(stcolor) legend(pos(6) rows(1) symysize(*0.5) size(small)) 
graph export "$path/graphs_paper/pelas_among_select_sectors.pdf",replace
restore
