***********************************************************************
* Project: The Demand for Government Debt – United States            *
* Description:                                                       *
* This script analyzes U.S. investment funds—Mutual Funds, ETFs, and *
* Closed-End Funds—as holders of government debt. It constructs and  *
* labels variables for regression-ready datasets, adjusts holdings   *
* for valuation changes, and estimates demand elasticities using     *
* PCA-identified monetary policy shocks. Also creates a time series  *
* graph and exports regression tables for the appendix.              *
***********************************************************************
clear all
set more off

/* Uncomment if running this file separately. 
* Set path

foreach path in "Add your directory here" "/Users/egemeneren/Dropbox/QT/Data/MaSc_RR_202507" "C:\Users\eg003762\BIS Dropbox\Egemen Eren\QT\Data\MaSc_RR_202507"  {
	capture cd "`path'"
	if _rc == 0 macro def path `path'
} 
*/


use "$path/generated_data/us_regression_cf_v202507.dta",clear

// Holding data
foreach x of varlist Mutualfunds ETFs Closedendfunds {
gen dln_`x'=ln(`x')-ln(l.`x')
gen ln_`x'=ln(`x')
}


///// Experimental

foreach i in 5 8 10 {
foreach x of varlist Mutualfunds ETFs Closedendfunds  {
gen val_ch`i'_`x'= l.`x'*val_ch`i'_pct
gen `x'_exval`i'=`x'-val_ch`i'_`x'
gen ln_`x'_exval`i'=ln(`x'_exval`i')
}
}
///////

/// Run this before regressions if creating tables. 
/// Label variables 

label var ln_Mutualfunds "ln(Mutual funds)"
label var ln_ETFs "ln(ETFs)"
label var ln_Closedendfunds "ln(Closed-ended funds)"
label var ln_Mutualfunds_exval8 "ln(Mutual funds)"
label var ln_ETFs_exval8 "ln(ETFs)"
label var ln_Closedendfunds_exval8 "ln(Closed-ended funds)"



/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////

//// Investment funds
keep if yq>tq(2004q2)

/// Graph

tw (line Mutualfunds yq, lcolor(black) lpattern(solid) lwidth(medthick)) (line ETFs yq, lcolor(black) lpattern(dash) lwidth(medthick)) ///
(line Closedendfunds yq, lcolor(black*0.6) lpattern(dash_dot) lwidth(medthick)) ///
, ytitle("$ trillions") xtitle(" ") graphre(color(white))  ylabel(,nogrid) ///
legend(rows(1) region(lwidth(none)) pos(bottom) col(5) size(*.9)  symxsize(*.4)) 
graph export "$path/graphs_paper/US_inv_funds.pdf",replace


// Table


global ln_controls "ln_GDP dln_GDP ir_inf_exc_fe ln_NBUS ln_VIXCLS ir_german_5y_zc yq yq_sq" 


ivreg2 ln_Mutualfunds_exval8 (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls ,robust bw(auto)  ffirst
 weakivtest,level(0.1)
local weak = `r(F_eff)'
local crt=`r(c_TSLS_10)'
local text = "addtext(Controls, \bf X and W, Sample, 2004q3-2024q2)"

local format = "label nor2  addstat(Anderson-Rubin Wald test (p-val), e(archi2p), Underidentification LM stat (p-val), e(idp)) adec(2) nonotes nocons sortvar(ir_THREEFY8) keep(ir_THREEFY8) "


local table = "$path/tables_paper/US_inv_funds_pca"
cap erase "`table'.tex"
cap erase "`table'.txt"

xi: ivreg2 ln_Mutualfunds_exval8 (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(4) `text'

xi: ivreg2 ln_ETFs_exval8 (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(4) `text'

xi: ivreg2 ln_Closedendfunds_exval8 (ir_THREEFY8=std_p1_jkbrwksx) $ln_controls , robust bw(auto)  ffirst
outreg2 using "`table'.tex", tex(frag) append `format' dec(4) `text'


cap erase "`table'.txt"

